﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/securitylake/SecurityLakeRequest.h>
#include <aws/securitylake/SecurityLake_EXPORTS.h>
#include <aws/securitylake/model/DataLakeConfiguration.h>
#include <aws/securitylake/model/Tag.h>

#include <utility>

namespace Aws {
namespace SecurityLake {
namespace Model {

/**
 */
class CreateDataLakeRequest : public SecurityLakeRequest {
 public:
  AWS_SECURITYLAKE_API CreateDataLakeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDataLake"; }

  AWS_SECURITYLAKE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Specify the Region or Regions that will contribute data to the rollup
   * region.</p>
   */
  inline const Aws::Vector<DataLakeConfiguration>& GetConfigurations() const { return m_configurations; }
  inline bool ConfigurationsHasBeenSet() const { return m_configurationsHasBeenSet; }
  template <typename ConfigurationsT = Aws::Vector<DataLakeConfiguration>>
  void SetConfigurations(ConfigurationsT&& value) {
    m_configurationsHasBeenSet = true;
    m_configurations = std::forward<ConfigurationsT>(value);
  }
  template <typename ConfigurationsT = Aws::Vector<DataLakeConfiguration>>
  CreateDataLakeRequest& WithConfigurations(ConfigurationsT&& value) {
    SetConfigurations(std::forward<ConfigurationsT>(value));
    return *this;
  }
  template <typename ConfigurationsT = DataLakeConfiguration>
  CreateDataLakeRequest& AddConfigurations(ConfigurationsT&& value) {
    m_configurationsHasBeenSet = true;
    m_configurations.emplace_back(std::forward<ConfigurationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) used to create and update the Glue table. This
   * table contains partitions generated by the ingestion and normalization of Amazon
   * Web Services log sources and custom sources.</p>
   */
  inline const Aws::String& GetMetaStoreManagerRoleArn() const { return m_metaStoreManagerRoleArn; }
  inline bool MetaStoreManagerRoleArnHasBeenSet() const { return m_metaStoreManagerRoleArnHasBeenSet; }
  template <typename MetaStoreManagerRoleArnT = Aws::String>
  void SetMetaStoreManagerRoleArn(MetaStoreManagerRoleArnT&& value) {
    m_metaStoreManagerRoleArnHasBeenSet = true;
    m_metaStoreManagerRoleArn = std::forward<MetaStoreManagerRoleArnT>(value);
  }
  template <typename MetaStoreManagerRoleArnT = Aws::String>
  CreateDataLakeRequest& WithMetaStoreManagerRoleArn(MetaStoreManagerRoleArnT&& value) {
    SetMetaStoreManagerRoleArn(std::forward<MetaStoreManagerRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of objects, one for each tag to associate with the data lake
   * configuration. For each tag, you must specify both a tag key and a tag value. A
   * tag value cannot be null, but it can be an empty string.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDataLakeRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDataLakeRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<DataLakeConfiguration> m_configurations;
  bool m_configurationsHasBeenSet = false;

  Aws::String m_metaStoreManagerRoleArn;
  bool m_metaStoreManagerRoleArnHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SecurityLake
}  // namespace Aws
