﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/Action.h>
#include <aws/glue/model/EventBatchingCondition.h>
#include <aws/glue/model/Predicate.h>
#include <aws/glue/model/TriggerType.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class CreateTriggerRequest : public GlueRequest {
 public:
  AWS_GLUE_API CreateTriggerRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTrigger"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the trigger.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateTriggerRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the workflow associated with the trigger.</p>
   */
  inline const Aws::String& GetWorkflowName() const { return m_workflowName; }
  inline bool WorkflowNameHasBeenSet() const { return m_workflowNameHasBeenSet; }
  template <typename WorkflowNameT = Aws::String>
  void SetWorkflowName(WorkflowNameT&& value) {
    m_workflowNameHasBeenSet = true;
    m_workflowName = std::forward<WorkflowNameT>(value);
  }
  template <typename WorkflowNameT = Aws::String>
  CreateTriggerRequest& WithWorkflowName(WorkflowNameT&& value) {
    SetWorkflowName(std::forward<WorkflowNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the new trigger.</p>
   */
  inline TriggerType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(TriggerType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline CreateTriggerRequest& WithType(TriggerType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A <code>cron</code> expression used to specify the schedule (see <a
   * href="https://docs.aws.amazon.com/glue/latest/dg/monitor-data-warehouse-schedule.html">Time-Based
   * Schedules for Jobs and Crawlers</a>. For example, to run something every day at
   * 12:15 UTC, you would specify: <code>cron(15 12 * * ? *)</code>.</p> <p>This
   * field is required when the trigger type is SCHEDULED.</p>
   */
  inline const Aws::String& GetSchedule() const { return m_schedule; }
  inline bool ScheduleHasBeenSet() const { return m_scheduleHasBeenSet; }
  template <typename ScheduleT = Aws::String>
  void SetSchedule(ScheduleT&& value) {
    m_scheduleHasBeenSet = true;
    m_schedule = std::forward<ScheduleT>(value);
  }
  template <typename ScheduleT = Aws::String>
  CreateTriggerRequest& WithSchedule(ScheduleT&& value) {
    SetSchedule(std::forward<ScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A predicate to specify when the new trigger should fire.</p> <p>This field is
   * required when the trigger type is <code>CONDITIONAL</code>.</p>
   */
  inline const Predicate& GetPredicate() const { return m_predicate; }
  inline bool PredicateHasBeenSet() const { return m_predicateHasBeenSet; }
  template <typename PredicateT = Predicate>
  void SetPredicate(PredicateT&& value) {
    m_predicateHasBeenSet = true;
    m_predicate = std::forward<PredicateT>(value);
  }
  template <typename PredicateT = Predicate>
  CreateTriggerRequest& WithPredicate(PredicateT&& value) {
    SetPredicate(std::forward<PredicateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The actions initiated by this trigger when it fires.</p>
   */
  inline const Aws::Vector<Action>& GetActions() const { return m_actions; }
  inline bool ActionsHasBeenSet() const { return m_actionsHasBeenSet; }
  template <typename ActionsT = Aws::Vector<Action>>
  void SetActions(ActionsT&& value) {
    m_actionsHasBeenSet = true;
    m_actions = std::forward<ActionsT>(value);
  }
  template <typename ActionsT = Aws::Vector<Action>>
  CreateTriggerRequest& WithActions(ActionsT&& value) {
    SetActions(std::forward<ActionsT>(value));
    return *this;
  }
  template <typename ActionsT = Action>
  CreateTriggerRequest& AddActions(ActionsT&& value) {
    m_actionsHasBeenSet = true;
    m_actions.emplace_back(std::forward<ActionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description of the new trigger.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateTriggerRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to <code>true</code> to start <code>SCHEDULED</code> and
   * <code>CONDITIONAL</code> triggers when created. True is not supported for
   * <code>ON_DEMAND</code> triggers.</p>
   */
  inline bool GetStartOnCreation() const { return m_startOnCreation; }
  inline bool StartOnCreationHasBeenSet() const { return m_startOnCreationHasBeenSet; }
  inline void SetStartOnCreation(bool value) {
    m_startOnCreationHasBeenSet = true;
    m_startOnCreation = value;
  }
  inline CreateTriggerRequest& WithStartOnCreation(bool value) {
    SetStartOnCreation(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to use with this trigger. You may use tags to limit access to the
   * trigger. For more information about tags in Glue, see <a
   * href="https://docs.aws.amazon.com/glue/latest/dg/monitor-tags.html">Amazon Web
   * Services Tags in Glue</a> in the developer guide. </p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateTriggerRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateTriggerRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Batch condition that must be met (specified number of events received or
   * batch time window expired) before EventBridge event trigger fires.</p>
   */
  inline const EventBatchingCondition& GetEventBatchingCondition() const { return m_eventBatchingCondition; }
  inline bool EventBatchingConditionHasBeenSet() const { return m_eventBatchingConditionHasBeenSet; }
  template <typename EventBatchingConditionT = EventBatchingCondition>
  void SetEventBatchingCondition(EventBatchingConditionT&& value) {
    m_eventBatchingConditionHasBeenSet = true;
    m_eventBatchingCondition = std::forward<EventBatchingConditionT>(value);
  }
  template <typename EventBatchingConditionT = EventBatchingCondition>
  CreateTriggerRequest& WithEventBatchingCondition(EventBatchingConditionT&& value) {
    SetEventBatchingCondition(std::forward<EventBatchingConditionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_workflowName;
  bool m_workflowNameHasBeenSet = false;

  TriggerType m_type{TriggerType::NOT_SET};
  bool m_typeHasBeenSet = false;

  Aws::String m_schedule;
  bool m_scheduleHasBeenSet = false;

  Predicate m_predicate;
  bool m_predicateHasBeenSet = false;

  Aws::Vector<Action> m_actions;
  bool m_actionsHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  bool m_startOnCreation{false};
  bool m_startOnCreationHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  EventBatchingCondition m_eventBatchingCondition;
  bool m_eventBatchingConditionHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
