/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.database.sourcemap;

import ghidra.program.database.sourcemap.SourceFileIdType;
import ghidra.util.BigEndianDataConverter;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.Instant;
import java.util.Arrays;
import java.util.HexFormat;
import org.apache.commons.lang3.StringUtils;

public final class SourceFile
implements Comparable<SourceFile> {
    private static final String FILE_SCHEME = "file";
    private static HexFormat hexFormat = HexFormat.of();
    private final String path;
    private final String filename;
    private final SourceFileIdType idType;
    private final byte[] identifier;
    private final int hash;
    private final String idDisplayString;

    public SourceFile(String path) {
        this(path, SourceFileIdType.NONE, null, true);
    }

    public SourceFile(String path, SourceFileIdType type, byte[] identifier) {
        this(path, type, identifier, true);
    }

    SourceFile(String pathToValidate, SourceFileIdType type, byte[] identifier, boolean validate) {
        block6: {
            if (validate) {
                if (StringUtils.isBlank((CharSequence)pathToValidate)) {
                    throw new IllegalArgumentException("pathToValidate cannot be null or blank");
                }
                try {
                    URI uri = new URI(FILE_SCHEME, null, pathToValidate, null).normalize();
                    this.path = uri.getPath();
                    if (this.path.endsWith("/")) {
                        throw new IllegalArgumentException("SourceFile URI must represent a file (not a directory)");
                    }
                    if (this.path.startsWith("/../")) {
                        throw new IllegalArgumentException("path must be absolute after normalization");
                    }
                    break block6;
                }
                catch (URISyntaxException e) {
                    throw new IllegalArgumentException("path not valid: " + e.getMessage());
                }
            }
            this.path = pathToValidate;
        }
        this.idType = type;
        this.filename = this.path.substring(this.path.lastIndexOf("/") + 1);
        this.identifier = this.validateAndCopyIdentifier(identifier);
        this.hash = this.computeHashcode();
        this.idDisplayString = this.computeIdDisplayString();
    }

    public URI getUri() {
        try {
            return new URI(FILE_SCHEME, null, this.path, null);
        }
        catch (URISyntaxException e) {
            throw new AssertionError((Object)"URISyntaxException on validated path");
        }
    }

    public String getPath() {
        return this.path;
    }

    public String getFilename() {
        return this.filename;
    }

    public SourceFileIdType getIdType() {
        return this.idType;
    }

    public byte[] getIdentifier() {
        byte[] copy = new byte[this.identifier.length];
        System.arraycopy(this.identifier, 0, copy, 0, this.identifier.length);
        return copy;
    }

    public int hashCode() {
        return this.hash;
    }

    public boolean equals(Object o) {
        if (!(o instanceof SourceFile)) {
            return false;
        }
        SourceFile otherFile = (SourceFile)o;
        if (!this.path.equals(otherFile.path)) {
            return false;
        }
        if (!this.idType.equals((Object)otherFile.idType)) {
            return false;
        }
        return Arrays.equals(this.identifier, otherFile.identifier);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.path);
        if (this.idType.equals((Object)SourceFileIdType.NONE)) {
            return sb.toString();
        }
        sb.append(" [");
        sb.append(this.idType.name());
        sb.append("=");
        sb.append(this.getIdAsString());
        sb.append("]");
        return sb.toString();
    }

    @Override
    public int compareTo(SourceFile sourceFile) {
        int comp = this.path.compareTo(sourceFile.path);
        if (comp != 0) {
            return comp;
        }
        comp = this.idType.compareTo(sourceFile.idType);
        if (comp != 0) {
            return comp;
        }
        return Arrays.compare(this.identifier, sourceFile.identifier);
    }

    public String getIdAsString() {
        return this.idDisplayString;
    }

    private int computeHashcode() {
        int result = this.path.hashCode();
        result = 31 * result + this.idType.hashCode();
        result = 31 * result + Arrays.hashCode(this.identifier);
        return result;
    }

    private byte[] validateAndCopyIdentifier(byte[] array) {
        if (array == null || this.idType == SourceFileIdType.NONE) {
            array = new byte[]{};
        }
        if (array.length > 64) {
            throw new IllegalArgumentException("identifier array too long; max is 64");
        }
        if (this.idType.getByteLength() != 0 && this.idType.getByteLength() != array.length) {
            throw new IllegalArgumentException("identifier array has wrong length for " + this.idType.name());
        }
        byte[] copy = new byte[array.length];
        System.arraycopy(array, 0, copy, 0, array.length);
        return copy;
    }

    private String computeIdDisplayString() {
        switch (this.idType) {
            case NONE: {
                return "";
            }
            case TIMESTAMP_64: {
                return Instant.ofEpochMilli(BigEndianDataConverter.INSTANCE.getLong(this.identifier)).toString();
            }
        }
        return hexFormat.formatHex(this.identifier);
    }
}

