package Lire::Config::DlfSchemaSpec;

use strict;

=pod

=head1 NAME

Lire::Config::DlfSchemaSpec - SelectSpec for DlfSchema.

=head1 DESCRIPTION

This Lire::Config::TypeSpec is a SelectSpec which defines an option
for each available DlfSchemas.

=head2 new( 'name' => $name, [ 'superservices' => 1 ], ... )

Creates a new DlfSchemaSpec object. If superservices is set to 1, only
superservices can be selected.

=cut

use base qw/ Lire::Config::SelectSpec /;

use Carp;

use Lire::DlfSchema;

sub new {
    my $self = shift->SUPER::new( @_ );
    my %args = @_;

    $self->superservices( $args{'superservices'} )
      if exists $args{'superservices'};

    return $self;
}

=pod

=head2 supservices( [ $new_value )

Returns (and optionnally change) the setting of the superservices
attributes. When this attribute is true, only superservices can be
selected by this object.

=cut

sub superservices {
    my $self = $_[0];

    $self->{'_superservices'} = $_[1] ? 1 : 0
      if @_ == 2;

    return $self->{'_superservices'};
}

sub is_valid {
    my ( $self, $val ) = @_;

    return !$self->required() unless $val;

    return Lire::DlfSchema->has_schema( $val );
}

sub normalize {
    my ( $self, $val ) = @_;

    return $self->is_valid( $val ) ? $val : undef;
}

sub add {
    croak( 'Lire::Config::DlfSchemaSpec cannot contain any components' );
}

sub options {
    my $self = $_[0];

    my @names = ( $self->{'_superservices'}
                  ? Lire::DlfSchema->superservices()
                  : Lire::DlfSchema->schemas() );
    my @options = ();
    foreach my $name ( sort @names ) {
        push @options,
          Lire::Config::DlfSchemaOptionSpec->new( 'name' => $name );
    }
    return @options;
}

package Lire::Config::DlfSchemaOptionSpec;

use base qw/Lire::Config::OptionSpec/;

use Lire::DlfSchema;

sub summary {
    return Lire::DlfSchema::load_schema( $_[0]->name() )->title();
}

sub description {
    return Lire::DlfSchema::load_schema( $_[0]->name() )->description();
}

1;

__END__

=pod

=head1 VERSION

$Id: DlfSchemaSpec.pm,v 1.10 2006/07/23 13:16:30 vanbaal Exp $

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
